<?xml version="1.0" encoding="utf-8"?>

<!-- 
  
  Things to handle regarding differences:
  
  1.  Have a parameter switch to turn difference reporting on or off.
  
  2.  Full body of the text with links from the changed content to the erratum/revision entry.
  
  3.  Formatting of the content that reflects the differences:
  ->Highlight with yellow
  ->Highlight with yellow
  ->Linethrough deletions and highlight with blue
  ->Nothing for content where differencing is off
  ->Hyperlink from a marker at the start of the differenced content to the relevant change.
  
  4.  List of revisions and errata.
  ->Summary information
  ->A list of links to the associated modified content.
  
  
-->

<xsl:transform xmlns:s="http://xbrl.org/specification/2007" xmlns:diff="http://xbrl.org/specification/2007/diff"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform" version="2.0">

  <xsl:output method="html" encoding="utf-8" doctype-public="-//W3C//DTD HTML 4.01 Transitional//EN" indent="no"/>

  <xsl:param name="locationTellTale"/>

  <!-- regex indicating what changeRefs to show.  type of @changeref attribute is xs:token, i.e. space separated -->
  <xsl:param name="show.changerefs">.*</xsl:param>

  <xsl:strip-space
    elements="
  s:p 
  s:br
  s:definition
  s:principle
  s:requirement
  s:long
  s:short
  s:item
  s:termref
  s:xtermref
  s:xmlref
  s:sectionref
  s:bibref
  s:exampleref
  s:figureref
  s:tableref
  s:errorref
  s:errorref2
  "/>

  <xsl:key name="ids" match="*[@id]" use="@id"/>
  <xsl:key name="refs"
    match="s:sectionref | s:requirementref | s:principleref | s:tableref | s:figureref | s:exampleref | s:termref | s:bibref | s:errorref | s:errorref2"
    use="@ref"/>

  <xsl:param name="generate-css" select="1"/>
  <xsl:param name="base.uri" select="'http://www.xbrl.org'"/>
  <xsl:param name="toc.level" select="7"/>

  <!-- 
    Parameter to control whether conformance test references 
    are included in produced documents.  It defaults to excluding
    them.
  -->
  <xsl:param name="include.conformance.tests" select="0"/>

  <!-- 
    Parameter to provide the URL of the document containing
    conformance suite information.
  -->
  <xsl:param name="conformance.map" select="'../../infrastructure/conformanceMap.xml'"/>
  <xsl:variable name="conformance.tests" select="document($conformance.map)/*/*"/>

  <!-- 
    Global variables
  -->

  <xsl:variable name="namespace-uri" select="'http://xbrl.org/specification/2007'"/>
  <xsl:variable name="specId" select="/s:spec/s:header/s:id"/>
  <xsl:variable name="pubType">
    <xsl:choose>
      <xsl:when test="/s:spec/s:header/s:pubtype!=''">
        <xsl:value-of select="/s:spec/s:header/s:pubtype"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:text>Specification</xsl:text>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:variable name="parentId" select="/s:spec/s:header/s:parentid"/>
  <xsl:variable name="projectFolder">
    <xsl:choose>
      <xsl:when test="$parentId!=''">
        <xsl:value-of select="$parentId"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$specId"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:variable name="status" select="/s:spec/s:header/s:status"/>
  <xsl:variable name="date" select="/s:spec/s:header/s:date"/>
  <xsl:variable name="erratarecdate" select="/s:spec/s:header/s:errataoriginaldate"/>
  <xsl:variable name="rootdir">
    <xsl:choose>
      <xsl:when test="$status='PWD'">
        <xsl:text>pub</xsl:text>
      </xsl:when>
      <xsl:when test="$status='CR'">
        <xsl:text>pub</xsl:text>
      </xsl:when>
      <xsl:when test="$status='PR'">
        <xsl:text>pub</xsl:text>
      </xsl:when>
      <xsl:when test="$status='REC'">
        <xsl:text>pub</xsl:text>
      </xsl:when>
      <xsl:when test="$status='WGN'">
        <xsl:text>pub</xsl:text>
      </xsl:when>
      <xsl:when test="$status='FWGN'">
        <xsl:text>pub</xsl:text>
      </xsl:when>
      <xsl:when test="$status='PER'">
        <xsl:text>pub</xsl:text>
      </xsl:when>
      <xsl:when test="$status='RR'">
        <xsl:text>pub</xsl:text>
      </xsl:when>
      <xsl:otherwise>
        <xsl:text>nopub</xsl:text>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:variable name="outputFileName">
    <xsl:choose>
      <xsl:when test="$pubType='Specification'">
        <xsl:choose>
          <!-- Specifications do not have "spec" in the file name -->
          <xsl:when test="$erratarecdate">
            <xsl:value-of
              select="concat($specId,'-',$status,'-',$erratarecdate/@year,'-',$erratarecdate/@month,'-',$erratarecdate/@day,'+corrected-errata-',$date/@year,'-',$date/@month,'-',$date/@day,'.html')"
            />
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="concat($specId,'-',$status,'-',$date/@year,'-',$date/@month,'-',$date/@day,'.html')"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:when>
      <xsl:otherwise>
        <!-- Everything else has the publication type code in the file name -->
        <xsl:value-of
          select="concat($specId,'-',$pubType,'-',$status,'-',$date/@year,'-',$date/@month,'-',$date/@day,'.html')"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:variable name="directory">
    <xsl:choose>
      <xsl:when test="$erratarecdate">
        <xsl:value-of
          select="concat($pubType,'/',$projectFolder,'/',$status,'-',$erratarecdate/@year,'-',$erratarecdate/@month,'-',$erratarecdate/@day,'/')"
        />
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of
          select="concat($pubType,'/',$projectFolder,'/',$status,'-',$date/@year,'-',$date/@month,'-',$date/@day,'/')"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <!-- 
    Create links to conformance suite tests 
  -->

  <xsl:template match="@id" mode="conformance">
    <xsl:if test="$include.conformance.tests">
      <xsl:variable name="id" select="."/>
      <xsl:variable name="related.tests" select="$conformance.tests[@partId=$id and @specId=$specId]"/>
      <xsl:for-each select="$related.tests">
        <span class="conformancetest"> &#xA0; <a href="../../{@testURL}">
            <xsl:value-of select="@number"/>
          </a>
        </span>
      </xsl:for-each>
    </xsl:if>
  </xsl:template>

  <!-- 
    Default handler to show elements that are not being treated correctly 
  -->
  <xsl:template match="*">
    <div class="error">
      <xsl:text>&lt;</xsl:text>
      <xsl:value-of select="name(.)"/>
      <xsl:text>&gt;</xsl:text>
      <xsl:text> </xsl:text>
      <xsl:apply-templates/>
      <xsl:text>&lt;/</xsl:text>
      <xsl:value-of select="name(.)"/>
      <xsl:text>&gt;</xsl:text>
    </div>
  </xsl:template>


  <!-- 
    Calculate the name of the output file and location where to put it
  -->
  <xsl:template match="/">
    <xsl:variable name="localDirectory" select="concat('../',$rootdir,'/',$directory)"/>
    <xsl:result-document href="{$localDirectory}{$outputFileName}">
      <xsl:apply-templates/>
    </xsl:result-document>
    <xsl:if test="$locationTellTale">
      <xsl:result-document href="location.txt" method="text">
        <xsl:value-of select="concat($localDirectory,$outputFileName)"/>
      </xsl:result-document>
    </xsl:if>
    <xsl:if test="$generate-css = 1">
      <xsl:result-document href="{$localDirectory}styles.css" method="text">
        <xsl:value-of select="document('styles.xml')/*"/>
      </xsl:result-document>
    </xsl:if>

  </xsl:template>

  <!-- ============================================================= -->
  <!-- Document structure and metadata templates -->
  <!-- ============================================================= -->

  <xsl:template match="s:spec">
    <html>
      <head>
        <title>
          <xsl:apply-templates select="s:header/s:title" mode="page-title"/>
          <xsl:if test="s:header/s:version">
            <xsl:call-template name="space"/>
            <xsl:apply-templates select="s:header/s:version" mode="page-title"/>
          </xsl:if>
        </title>
        <link rel="stylesheet" type="text/css" href="styles.css"/>
        <xsl:call-template name="script"/>
      </head>
      <body>
        <xsl:apply-templates/>
        <xsl:call-template name="endnotes"/>
      </body>
    </html>
  </xsl:template>

  <xsl:template match="s:header">
    <div class="head">
      <xsl:call-template name="banner"/>
      <xsl:call-template name="newline"/>
      <xsl:apply-templates select="s:title"/>
      <xsl:apply-templates select="s:subtitle"/>
      <xsl:call-template name="newline"/>
      <h2>
        <xsl:call-template name="status-text"/>
        <xsl:call-template name="space"/>
        <xsl:if test="s:errataoriginaldate">
          <xsl:call-template name="full-erratarecdate"/>
          <xsl:text> with errata corrections to </xsl:text>
        </xsl:if>
        <xsl:call-template name="full-date"/>
      </h2>

      <p>
        <xsl:text>Copyright &#169;</xsl:text>
        <xsl:value-of select="s:date/@year"/>
        <xsl:text> XBRL International Inc., All Rights Reserved.</xsl:text>
      </p>

      <dl>
        <xsl:call-template name="url"/>
        <xsl:apply-templates select="s:authors"/>
        <xsl:apply-templates select="s:contributors"/>
      </dl>
      <hr/>
      <xsl:call-template name="status"/>
      <xsl:apply-templates select="s:notice"/>
      <xsl:apply-templates select="s:abstract"/>
    </div>
  </xsl:template>

  <xsl:template match="s:title">
    <h1>
      <xsl:call-template name="anchor">
        <xsl:with-param name="node" select="."/>
        <xsl:with-param name="conditional" select="0"/>
        <xsl:with-param name="default.id" select="'title'"/>
      </xsl:call-template>
      <xsl:apply-templates/>
      <xsl:call-template name="space"/>
      <xsl:value-of select="../s:version"/>
    </h1>
  </xsl:template>

  <xsl:template match="s:title" mode="page-title">
    <xsl:apply-templates/>
  </xsl:template>

  <xsl:template match="s:version" mode="page-title">
    <xsl:apply-templates/>
  </xsl:template>


  <xsl:template match="s:subtitle">
    <xsl:call-template name="newline"/>
    <h2>
      <xsl:call-template name="anchor">
        <xsl:with-param name="node" select="."/>
        <xsl:with-param name="conditional" select="0"/>
        <xsl:with-param name="default.id" select="'subtitle'"/>
      </xsl:call-template>
      <xsl:apply-templates/>
    </h2>
  </xsl:template>

  <xsl:template name="url">
    <xsl:choose>
      <xsl:when test="$rootdir='pub'">
        <xsl:call-template name="publishURL"/>
      </xsl:when>
      <xsl:otherwise>
        <dt>This version:</dt>
        <dd>
          <xsl:value-of select="$outputFileName"/>
        </dd>
      </xsl:otherwise>
    </xsl:choose>

  </xsl:template>

  <xsl:template name="publishURL">
    <xsl:variable name="pubURL" select="concat($base.uri,'/',$directory,$outputFileName)"/>
    <dt>This version:</dt>
    <dd>
      <a href="{$pubURL}">
        <xsl:text>&lt;</xsl:text>
        <xsl:value-of select="$pubURL"/>
        <xsl:text>&gt;</xsl:text>
      </a>
    </dd>
  </xsl:template>

  <xsl:template match="s:authors">
    <dt>
      <xsl:text>Editor</xsl:text>
      <xsl:if test="count(s:person) > 1">
        <xsl:text>s</xsl:text>
      </xsl:if>
      <xsl:text>:</xsl:text>
    </dt>
    <xsl:apply-templates select="s:person" mode="this_document"/>
  </xsl:template>

  <xsl:template match="s:contributors">
    <dt>
      <xsl:text>Contributor</xsl:text>
      <xsl:if test="count(s:person) > 1">
        <xsl:text>s</xsl:text>
      </xsl:if>
      <xsl:text>:</xsl:text>
    </dt>
    <xsl:apply-templates select="s:person" mode="this_document"/>
  </xsl:template>

  <xsl:template match="s:person" mode="this_document">
    <dd>
      <a name="{@id}" id="{@id}"/>
      <xsl:apply-templates mode="this_document"/>
    </dd>
  </xsl:template>

  <xsl:template match="s:name" mode="this_document">
    <xsl:apply-templates/>, </xsl:template>

  <xsl:template match="s:affiliation" mode="this_document">
    <xsl:text> </xsl:text>
    <xsl:apply-templates/>
  </xsl:template>

  <xsl:template match="s:email" mode="this_document">
    <xsl:text> </xsl:text>
    <a href="mailto:{text()}">
      <xsl:text>&lt;</xsl:text><xsl:apply-templates/><xsl:text>&gt;</xsl:text>
    </a>
  </xsl:template>

  <xsl:template match="s:person">
    <dd>
      <a name="{@id}" id="{@id}"/>
      <xsl:apply-templates/>
    </dd>
  </xsl:template>

  <xsl:template match="s:name">
    <xsl:apply-templates/>
  </xsl:template>

  <xsl:template match="s:affiliation">
    <xsl:text> </xsl:text>
    <xsl:apply-templates/>
  </xsl:template>

  <xsl:template match="s:email">
    <xsl:text> </xsl:text>
    <a href="mailto:{text()}">
      <xsl:text>&lt;</xsl:text>
      <xsl:apply-templates/>
      <xsl:text>&gt;</xsl:text>
    </a>
  </xsl:template>

  <xsl:template match="s:notice">
    <div class="notice">
      <p class="prefix">
        <b>NOTICE:</b>
      </p>
      <xsl:apply-templates/>
    </div>
  </xsl:template>

  <xsl:template match="s:abstract">
    <div>
      <xsl:call-template name="newline"/>
      <h2>
        <xsl:call-template name="anchor">
          <xsl:with-param name="conditional" select="0"/>
          <xsl:with-param name="default.id" select="'abstract'"/>
        </xsl:call-template>
        <xsl:text>Abstract</xsl:text>
      </h2>
      <xsl:apply-templates/>
    </div>

    <xsl:variable name="comments" select="//s:comment"/>
    <xsl:if test="$comments">
      <h3>
        <xsl:call-template name="anchor">
          <xsl:with-param name="conditional" select="0"/>
          <xsl:with-param name="default.id" select="'comments'"/>
        </xsl:call-template>
        <xsl:text>Comment</xsl:text>
        <xsl:if test="count($comments) > 1">
          <xsl:text>s</xsl:text>
        </xsl:if>
      </h3>
      <p class="toc">
        <xsl:apply-templates select="$comments" mode="toc"/>
      </p>
    </xsl:if>

  </xsl:template>

  <xsl:template match="s:body">
    <xsl:call-template name="toc"/>
    <div class="body">
      <xsl:apply-templates/>
    </div>
  </xsl:template>

  <xsl:template match="s:section">
    <xsl:variable name="depth" select="count(ancestor::s:section)+1"/>
    <div class="{concat('div',$depth)}">
      <xsl:call-template name="anchor">
        <xsl:with-param name="conditional" select="0"/>
      </xsl:call-template>
      <xsl:apply-templates/>
    </div>
  </xsl:template>

  <xsl:template match="s:head">
    <xsl:variable name="depth" select="count(ancestor::s:section)"/>
    <xsl:choose>
      <xsl:when test="$depth &lt; 6">
        <xsl:element name="{concat('h',$depth)}">
          <xsl:apply-templates select=".." mode="divnum"/>
          <xsl:call-template name="space"/>
          <xsl:apply-templates/>
          <xsl:apply-templates select="../@id" mode="conformance"/>
        </xsl:element>
      </xsl:when>
      <xsl:otherwise>
        <xsl:element name="h6">
          <xsl:apply-templates select=".." mode="divnum"/>
          <xsl:call-template name="space"/>
          <xsl:apply-templates/>
          <xsl:apply-templates select="../@id" mode="conformance"/>
        </xsl:element>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="s:back">
    <div class="back">
      <xsl:apply-templates/>
    </div>
  </xsl:template>

  <!-- ============================================================= -->
  <!-- Table of contents -->
  <!-- ============================================================= -->

  <xsl:template name="toc">
    <xsl:if test="$toc.level &gt; 0">
      <div class="toc">
        <xsl:call-template name="newline"/>
        <h2>
          <xsl:call-template name="anchor">
            <xsl:with-param name="conditional" select="0"/>
            <xsl:with-param name="default.id" select="'contents'"/>
          </xsl:call-template>
          <xsl:text>Table of Contents</xsl:text>
        </h2>
        <p class="toc">
          <xsl:apply-templates select="s:section" mode="toc"/>
        </p>
        <xsl:if test="../s:back">
          <xsl:call-template name="newline"/>
          <h3>
            <xsl:call-template name="anchor">
              <xsl:with-param name="conditional" select="0"/>
              <xsl:with-param name="default.id" select="'appendices'"/>
            </xsl:call-template>
            <xsl:text>Appendi</xsl:text>
            <xsl:choose>
              <xsl:when test="count(../s:back/s:section) > 1">
                <xsl:text>ces</xsl:text>
              </xsl:when>
              <xsl:otherwise>
                <xsl:text>x</xsl:text>
              </xsl:otherwise>
            </xsl:choose>
          </h3>
          <p class="toc">
            <xsl:apply-templates mode="toc" select="../s:back/s:section"/>
          </p>
        </xsl:if>

        <xsl:variable name="endnotes" select="//s:footnote[not(ancestor::s:table)]"/>
        <xsl:if test="$endnotes">
          <p class="toc">
            <a href="#endnotes">
              <xsl:text>End note</xsl:text>
              <xsl:if test="count($endnotes) > 1">
                <xsl:text>s</xsl:text>
              </xsl:if>
            </a>
          </p>
        </xsl:if>

        <xsl:variable name="tables" select="//s:caption[parent::s:table]"/>
        <xsl:if test="$tables">
          <h3>
            <xsl:call-template name="anchor">
              <xsl:with-param name="conditional" select="0"/>
              <xsl:with-param name="default.id" select="'examples'"/>
            </xsl:call-template>
            <xsl:text>Table</xsl:text>
            <xsl:if test="count($tables) > 1">
              <xsl:text>s</xsl:text>
            </xsl:if>
          </h3>
          <p class="toc">
            <xsl:apply-templates select="$tables" mode="toc"/>
          </p>
        </xsl:if>

        <xsl:variable name="figures" select="//s:caption[parent::s:figure]"/>
        <xsl:if test="$figures">
          <h3>
            <xsl:call-template name="anchor">
              <xsl:with-param name="conditional" select="0"/>
              <xsl:with-param name="default.id" select="'figures'"/>
            </xsl:call-template>
            <xsl:text>Figure</xsl:text>
            <xsl:if test="count($figures) > 1">
              <xsl:text>s</xsl:text>
            </xsl:if>
          </h3>
          <p class="toc">
            <xsl:apply-templates select="$figures" mode="toc"/>
          </p>
        </xsl:if>

        <xsl:variable name="examples" select="//s:caption[parent::s:example]"/>
        <xsl:if test="$examples">
          <h3>
            <xsl:call-template name="anchor">
              <xsl:with-param name="conditional" select="0"/>
              <xsl:with-param name="default.id" select="'figures'"/>
            </xsl:call-template>
            <xsl:text>Example</xsl:text>
            <xsl:if test="count($examples) > 1">
              <xsl:text>s</xsl:text>
            </xsl:if>
          </h3>
          <p class="toc">
            <xsl:apply-templates select="$examples" mode="toc"/>
          </p>
        </xsl:if>

        <xsl:variable name="definitions" select="//s:definition[(count(@noindex)=0 or @noindex=false()) and empty(ancestor::s:verbatim)]"/>
        <xsl:if test="$definitions">
          <xsl:call-template name="newline"/>
          <h3>
            <xsl:call-template name="anchor">
              <xsl:with-param name="conditional" select="0"/>
              <xsl:with-param name="default.id" select="'definitions'"/>
            </xsl:call-template>
            <xsl:text>Definition</xsl:text>
            <xsl:if test="count($definitions) > 1">
              <xsl:text>s</xsl:text>
            </xsl:if>
          </h3>
          <p class="toc">
            <xsl:apply-templates mode="toc" select="$definitions">
              <xsl:sort select="@term" data-type="text" order="ascending"/>
            </xsl:apply-templates>
          </p>
        </xsl:if>

        <xsl:variable name="principles" select="//s:principle[count(@noindex)=0 or @noindex=false()]"/>
        <xsl:if test="$principles">
          <xsl:call-template name="newline"/>
          <h3>
            <xsl:call-template name="anchor">
              <xsl:with-param name="conditional" select="0"/>
              <xsl:with-param name="default.id" select="'principles'"/>
            </xsl:call-template>
            <xsl:text>Principle</xsl:text>
            <xsl:if test="count($principles) > 1">
              <xsl:text>s</xsl:text>
            </xsl:if>
          </h3>
          <p class="toc">
            <xsl:apply-templates mode="toc" select="$principles">
              <xsl:sort select="@name" data-type="text" order="ascending"/>
            </xsl:apply-templates>
          </p>
        </xsl:if>

        <xsl:variable name="requirements" select="//s:requirement[count(@noindex)=0 or @noindex=false()]"/>
        <xsl:if test="$requirements">
          <xsl:call-template name="newline"/>
          <h3>
            <xsl:call-template name="anchor">
              <xsl:with-param name="conditional" select="0"/>
              <xsl:with-param name="default.id" select="'requirements'"/>
            </xsl:call-template>
            <xsl:text>Requirement</xsl:text>
            <xsl:if test="count($requirements) > 1">
              <xsl:text>s</xsl:text>
            </xsl:if>
          </h3>
          <p class="toc">
            <xsl:apply-templates mode="toc" select="$requirements">
              <xsl:sort select="@name" data-type="text" order="ascending"/>
            </xsl:apply-templates>
          </p>
        </xsl:if>

        <xsl:variable name="errors" select="//s:error"/>
        <xsl:variable name="errors2" select="//s:error2"/>
        <xsl:if test="$errors">
          <xsl:call-template name="newline"/>
          <h3>
            <xsl:call-template name="anchor">
              <xsl:with-param name="conditional" select="0"/>
              <xsl:with-param name="default.id" select="'error_codes'"/>
            </xsl:call-template>
            <xsl:text>Error code</xsl:text>
            <xsl:if test="count($errors) + count($errors2) > 1">
              <xsl:text>s</xsl:text>
            </xsl:if>
          </h3>
          <p class="toc">
            <xsl:apply-templates mode="toc" select="$errors">
              <xsl:sort select="s:errorcode/@code" data-type="text" order="ascending"/>
            </xsl:apply-templates>
          </p>
        </xsl:if>
        <xsl:if test="$errors2">

          <p class="toc">
            <xsl:apply-templates mode="toc" select="$errors2">
              <xsl:sort select="s:errorcode2/@code" data-type="text" order="ascending"/>
            </xsl:apply-templates>
          </p>
        </xsl:if>

      </div>

      <hr/>
    </xsl:if>

  </xsl:template>

  <xsl:template mode="toc" match="s:section">
    <xsl:call-template name="indent"/>
    <xsl:apply-templates select="." mode="divnum-toc"/>
    <xsl:call-template name="space"/>
    <a>
      <xsl:attribute name="href">
        <xsl:call-template name="href.target">
          <xsl:with-param name="target" select="."/>
        </xsl:call-template>
      </xsl:attribute>
      <xsl:apply-templates select="s:head" mode="text"/>
    </a>
    <br/>
    <xsl:call-template name="newline"/>
    <xsl:if test="count(ancestor::*) &lt;= $toc.level">
      <xsl:apply-templates select="s:section" mode="toc"/>
    </xsl:if>
  </xsl:template>

  <xsl:template name="indent">
    <xsl:for-each select="ancestor::s:section">
      <xsl:call-template name="space"/>
      <xsl:call-template name="space"/>
      <xsl:call-template name="space"/>
    </xsl:for-each>
  </xsl:template>

  <xsl:template name="href.target">
    <xsl:param name="target" select="."/>
    <xsl:text>#</xsl:text>
    <xsl:choose>
      <xsl:when test="$target/@id">
        <xsl:value-of select="$target/@id"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="generate-id($target)"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template mode="divnum" match="s:section">
    <xsl:param name="depth" select="count(ancestor::s:section)+1"/>
    <xsl:choose>
      <xsl:when test="ancestor::s:body">
        <xsl:choose>
          <xsl:when test="$depth = 1">
            <xsl:number format="1"/>
          </xsl:when>
          <xsl:when test="$depth = 2">
            <xsl:number level="multiple" format="1.1"/>
          </xsl:when>
          <xsl:when test="$depth = 3">
            <xsl:number level="multiple" format="1.1.1"/>
          </xsl:when>
          <xsl:when test="$depth = 4">
            <xsl:number level="multiple" format="1.1.1.1"/>
          </xsl:when>
          <xsl:when test="$depth = 5">
            <xsl:number level="multiple" format="1.1.1.1.1"/>
          </xsl:when>
          <xsl:when test="$depth = 6">
            <xsl:number level="multiple" format="1.1.1.1.1.1"/>
          </xsl:when>
          <xsl:when test="$depth = 7">
            <xsl:number level="multiple" format="1.1.1.1.1.1.1"/>
          </xsl:when>
          <xsl:when test="$depth = 8">
            <xsl:number level="multiple" format="1.1.1.1.1.1.1.1"/>
          </xsl:when>
        </xsl:choose>
      </xsl:when>
      <xsl:otherwise>
        <xsl:choose>
          <xsl:when test="$depth = 1">
            <xsl:text>Appendix </xsl:text>
            <xsl:number format="A"/>
          </xsl:when>
          <xsl:when test="$depth = 2">
            <xsl:number level="multiple" format="A.1"/>
          </xsl:when>
          <xsl:when test="$depth = 3">
            <xsl:number level="multiple" format="A.1.1"/>
          </xsl:when>
          <xsl:when test="$depth = 4">
            <xsl:number level="multiple" format="A.1.1.1"/>
          </xsl:when>
          <xsl:when test="$depth = 5">
            <xsl:number level="multiple" format="A.1.1.1.1"/>
          </xsl:when>
          <xsl:when test="$depth = 6">
            <xsl:number level="multiple" format="A.1.1.1.1.1"/>
          </xsl:when>
          <xsl:when test="$depth = 7">
            <xsl:number level="multiple" format="A.1.1.1.1.1.1"/>
          </xsl:when>
          <xsl:when test="$depth = 8">
            <xsl:number level="multiple" format="A.1.1.1.1.1.1.1"/>
          </xsl:when>
        </xsl:choose>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template mode="divnum-toc" match="s:section">
    <xsl:param name="depth" select="count(ancestor::s:section)+1"/>
    <xsl:choose>
      <xsl:when test="ancestor::s:body">
        <xsl:choose>
          <xsl:when test="$depth = 1">
            <xsl:number format="1"/>
          </xsl:when>
          <xsl:when test="$depth = 2">
            <xsl:number level="multiple" format="1.1"/>
          </xsl:when>
          <xsl:when test="$depth = 3">
            <xsl:number level="multiple" format="1.1.1"/>
          </xsl:when>
          <xsl:when test="$depth = 4">
            <xsl:number level="multiple" format="1.1.1.1"/>
          </xsl:when>
          <xsl:when test="$depth = 5">
            <xsl:number level="multiple" format="1.1.1.1.1"/>
          </xsl:when>
          <xsl:when test="$depth = 6">
            <xsl:number level="multiple" format="1.1.1.1.1.1"/>
          </xsl:when>
          <xsl:when test="$depth = 7">
            <xsl:number level="multiple" format="1.1.1.1.1.1.1"/>
          </xsl:when>
          <xsl:when test="$depth = 8">
            <xsl:number level="multiple" format="1.1.1.1.1.1.1.1"/>
          </xsl:when>
        </xsl:choose>
      </xsl:when>
      <xsl:otherwise>
        <xsl:choose>
          <xsl:when test="$depth = 1">
            <xsl:text/>
            <xsl:number format="A"/>
          </xsl:when>
          <xsl:when test="$depth = 2">
            <xsl:number level="multiple" format="A.1"/>
          </xsl:when>
          <xsl:when test="$depth = 3">
            <xsl:number level="multiple" format="A.1.1"/>
          </xsl:when>
          <xsl:when test="$depth = 4">
            <xsl:number level="multiple" format="A.1.1.1"/>
          </xsl:when>
          <xsl:when test="$depth = 5">
            <xsl:number level="multiple" format="A.1.1.1.1"/>
          </xsl:when>
          <xsl:when test="$depth = 6">
            <xsl:number level="multiple" format="A.1.1.1.1.1"/>
          </xsl:when>
          <xsl:when test="$depth = 7">
            <xsl:number level="multiple" format="A.1.1.1.1.1.1"/>
          </xsl:when>
          <xsl:when test="$depth = 8">
            <xsl:number level="multiple" format="A.1.1.1.1.1.1.1"/>
          </xsl:when>
        </xsl:choose>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="s:head" mode="text">
    <xsl:apply-templates/>
  </xsl:template>

  <xsl:template mode="toc" match="s:caption[parent::s:table]">
    <xsl:value-of select="count(preceding::s:caption[parent::s:table])+1"/>
    <xsl:call-template name="space"/>
    <a>
      <xsl:attribute name="href">
        <xsl:call-template name="href.target">
          <xsl:with-param name="target" select=".."/>
        </xsl:call-template>
      </xsl:attribute>
      <xsl:apply-templates/>
    </a>
    <br/>
  </xsl:template>

  <xsl:template mode="toc" match="s:caption[parent::s:example]">
    <xsl:value-of select="count(preceding::s:caption[parent::s:example])+1"/>
    <xsl:call-template name="space"/>
    <a>
      <xsl:attribute name="href">
        <xsl:call-template name="href.target">
          <xsl:with-param name="target" select=".."/>
        </xsl:call-template>
      </xsl:attribute>
      <xsl:apply-templates/>
    </a>
    <br/>
  </xsl:template>

  <xsl:template mode="toc" match="s:caption[parent::s:figure]">
    <xsl:value-of select="count(preceding::s:caption[parent::s:figure])+1"/>
    <xsl:call-template name="space"/>
    <a>
      <xsl:attribute name="href">
        <xsl:call-template name="href.target">
          <xsl:with-param name="target" select=".."/>
        </xsl:call-template>
      </xsl:attribute>
      <xsl:apply-templates/>
    </a>
    <br/>
  </xsl:template>

  <xsl:template mode="toc" match="s:error">
    <a>
      <xsl:attribute name="href">
        <xsl:call-template name="href.target">
          <xsl:with-param name="target" select="."/>
        </xsl:call-template>
      </xsl:attribute>
      <xsl:apply-templates select="s:errorcode"/>
    </a>
    <br/>
  </xsl:template>
  <xsl:template mode="toc" match="s:error2">
    <a>
      <xsl:attribute name="href">
        <xsl:call-template name="href.target">
          <xsl:with-param name="target" select="."/>
        </xsl:call-template>
      </xsl:attribute>
      <xsl:apply-templates select="s:errorcode2"/>
    </a>
    <br/>
  </xsl:template>

  <xsl:template mode="toc" match="s:definition">
    <a>
      <xsl:attribute name="href">
        <xsl:call-template name="href.target">
          <xsl:with-param name="target" select="."/>
        </xsl:call-template>
      </xsl:attribute>
      <xsl:value-of select="@term"/>
    </a>
    <br/>
  </xsl:template>

  <xsl:template mode="toc" match="s:principle">
    <a>
      <xsl:attribute name="href">
        <xsl:call-template name="href.target">
          <xsl:with-param name="target" select="."/>
        </xsl:call-template>
      </xsl:attribute>
      <xsl:value-of select="@name"/>: <xsl:apply-templates select="s:short" mode="toc"/>
    </a>
    <br/>
  </xsl:template>

  <xsl:template mode="toc" match="s:requirement">
    <a>
      <xsl:attribute name="href">
        <xsl:call-template name="href.target">
          <xsl:with-param name="target" select="."/>
        </xsl:call-template>
      </xsl:attribute>
      <xsl:value-of select="@name"/>: <xsl:apply-templates select="s:short" mode="toc"/>
    </a>
    <br/>
  </xsl:template>

  <xsl:template mode="toc" match="s:comment">
    <xsl:value-of select="count(preceding::s:comment)+1"/>
    <xsl:call-template name="space"/>
    <a>
      <xsl:attribute name="href">
        <xsl:call-template name="href.target">
          <xsl:with-param name="target" select="."/>
        </xsl:call-template>
      </xsl:attribute>
      <xsl:apply-templates select="key('ids', @by)/s:name"/>
    </a>
    <xsl:text>:</xsl:text>
    <xsl:apply-templates/>
    <br/>
  </xsl:template>

  <!-- ============================================================= -->
  <!-- Chunk templates -->
  <!-- ============================================================= -->

  <xsl:template match="s:p">
    <p>
      <xsl:if test="@id">
        <xsl:attribute name="id">
          <xsl:value-of select="@id"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@role">
        <xsl:attribute name="class">
          <xsl:value-of select="@role"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:call-template name="if-diff"/>
      <xsl:apply-templates/>
    </p>
  </xsl:template>



  <xsl:template match="s:p[s:verbatim]">
    <div>
      <xsl:if test="@id">
        <xsl:attribute name="id">
          <xsl:value-of select="@id"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@role">
        <xsl:attribute name="class">
          <xsl:value-of select="@role"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:apply-templates/>
    </div>
  </xsl:template>


  <xsl:template match="s:p[ancestor::s:revision]">
    <p class="revision">
      <xsl:if test="@id">
        <xsl:attribute name="id">
          <xsl:value-of select="@id"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:apply-templates/>
    </p>
  </xsl:template>

  <xsl:template match="s:p[ancestor::s:errata]">
    <p class="errata">
      <xsl:if test="@id">
        <xsl:attribute name="id">
          <xsl:value-of select="@id"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:apply-templates/>
    </p>
  </xsl:template>

  <xsl:template match="s:p[s:code and count(text())=0]">
    <p class="code-paragraph">
      <xsl:if test="@id">
        <xsl:attribute name="id">
          <xsl:value-of select="@id"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@role">
        <xsl:attribute name="class">
          <xsl:value-of select="@role"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:apply-templates/>
    </p>
  </xsl:template>


  <xsl:template match="s:comment">
    <span class="comment">
      <xsl:if test="@id">
        <xsl:attribute name="id">
          <xsl:value-of select="@id"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@role">
        <xsl:attribute name="class">
          <xsl:value-of select="@role"/>
        </xsl:attribute>
      </xsl:if>
      <a>
        <xsl:attribute name="name" select="generate-id()"/>
        <xsl:attribute name="id" select="generate-id()"/>
        <!-- xsl:attribute name="id">
          <xsl:call-template name="href.target">
          <xsl:with-param name="target" select="."/>
          </xsl:call-template>
          </xsl:attribute-->
      </a>
      <xsl:text>[</xsl:text>
      <xsl:apply-templates select="key('ids', @by)/s:name"/>
      <xsl:text>: </xsl:text>
      <xsl:apply-templates/>
      <xsl:text>]</xsl:text>
    </span>
  </xsl:template>

  <xsl:template match="s:fileExample">
    <xsl:apply-templates select="s:caption" mode="fileExample.heading"/>
    <xsl:choose>
      <xsl:when test="@role">
        <div class="{@role}">
          <xsl:apply-templates/>
        </div>
      </xsl:when>
      <xsl:otherwise>
        <div class="outerExample">
          <xsl:apply-templates/>
        </div>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>


  <xsl:template match="s:example">
    <xsl:apply-templates select="s:caption" mode="example.heading"/>
    <xsl:choose>
      <xsl:when test="@role">
        <div class="{@role}">
          <xsl:apply-templates/>
        </div>
      </xsl:when>
      <xsl:otherwise>
        <div class="outerExample">
          <xsl:apply-templates/>
        </div>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="s:caption[parent::s:example]"/>

  <xsl:template match="s:caption[parent::s:fileExample]"/>

  <xsl:template match="s:caption" mode="fileExample.heading">
    <div class="nonChunkHeading">
      <a name="{../@id}" id="{../@id}"/>
      <p>
        <xsl:text>Schema file </xsl:text>
        <xsl:value-of select="count(preceding::s:caption[parent::s:fileExample])+1"/>
        <xsl:text>: </xsl:text>
        <xsl:apply-templates/>
      </p>
      <xsl:apply-templates select="../@id" mode="conformance"/>
    </div>
  </xsl:template>

  <xsl:template match="s:caption" mode="example.heading">
    <div class="chunkHeading">
      <a name="{../@id}" id="{../@id}"/>
      <p>
        <xsl:call-template name="if-diff"/>
        <xsl:text>Example </xsl:text>
        <xsl:value-of select="count(preceding::s:caption[parent::s:example])+1"/>
        <xsl:text>: </xsl:text>
        <xsl:apply-templates/>
      </p>
      <xsl:apply-templates select="../@id" mode="conformance"/>
    </div>
  </xsl:template>

  <xsl:template match="s:figure">
    <xsl:apply-templates select="s:caption" mode="figure.heading"/>
    <xsl:choose>
      <xsl:when test="@role">
        <div class="{@role}">
          <xsl:apply-templates/>
        </div>
      </xsl:when>
      <xsl:otherwise>
        <div class="figure">
          <xsl:apply-templates/>
        </div>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="s:caption[parent::s:figure]"/>

  <xsl:template match="s:caption" mode="figure.heading">
    <div class="chunkHeading">
      <a name="{../@id}" id="{../@id}"/>
      <xsl:text>Figure </xsl:text>
      <xsl:value-of select="count(preceding::s:caption[parent::s:figure])+1"/>
      <xsl:text>: </xsl:text>
      <xsl:apply-templates/>
      <xsl:apply-templates select="../@id" mode="conformance"/>
    </div>
  </xsl:template>

  <xsl:template match="s:img">
    <img src="{@src}">
      <xsl:if test="@alt">

        <xsl:attribute name="alt">
          <xsl:value-of select="@alt"/>
        </xsl:attribute>
      </xsl:if>
    </img>
  </xsl:template>






  <xsl:template match="s:verbatim">
    <xsl:apply-templates mode="verbatim"/>
  </xsl:template>

  <!-- ============================================================= -->
  <!-- Phrase templates -->
  <!-- ============================================================= -->

  <xsl:template match="s:code">
    <code>
      <xsl:apply-templates/>
    </code>
  </xsl:template>

  <xsl:template match="s:elt">
    <code class="element"> &lt;<xsl:apply-templates/>&gt; </code>
  </xsl:template>

  <xsl:template match="s:att">
    <code class="attribute"> @<xsl:apply-templates/>
    </code>
  </xsl:template>

  <xsl:template match="s:val">
    <code class="value">
      <xsl:apply-templates/>
    </code>
  </xsl:template>

  <xsl:template match="s:var">
    <code class="variable">
      <xsl:apply-templates/>
    </code>
  </xsl:template>

  <xsl:template match="s:rfc2119">
    <strong>
      <xsl:apply-templates/>
    </strong>
  </xsl:template>

  <xsl:template match="s:em">
    <em>
      <xsl:apply-templates/>
    </em>
  </xsl:template>

  <xsl:template match="s:strong">
    <strong>
      <xsl:apply-templates/>
    </strong>
  </xsl:template>

  <xsl:template match="s:center">
    <center>
      <xsl:apply-templates/>
    </center>
  </xsl:template>

  <xsl:template match="s:superscript">
    <sup>
      <xsl:apply-templates/>
    </sup>
  </xsl:template>

  <xsl:template match="s:subscript">
    <sub>
      <xsl:apply-templates/>
    </sub>
  </xsl:template>

  <xsl:template match="s:phrase">
    <span>
      <xsl:call-template name="add-id"/>
      <xsl:call-template name="if-diff"/>
      <xsl:apply-templates/>
    </span>
  </xsl:template>

  <xsl:template match="s:nbsp">
    <xsl:call-template name="nbsp"/>
  </xsl:template>

  <xsl:template match="s:br">
    <xsl:element name="br">
      <xsl:call-template name="if-diff"/>
    </xsl:element>
  </xsl:template>


  <xsl:template match="s:amp">
    <xsl:text>&amp;</xsl:text>
  </xsl:template>

  <xsl:template match="s:definition">
    <span class="definition">
      <xsl:call-template name="if-diff"/>
      <a name="{@id}" id="{@id}" title="{@term}"/>
      <xsl:apply-templates/>
      <xsl:apply-templates select="@id" mode="conformance"/>
    </span>
  </xsl:template>

  <xsl:template match="s:term">
    <span class="term">
      <xsl:call-template name="if-diff"/>
      <xsl:apply-templates/>
    </span>
  </xsl:template>

  <xsl:template match="s:tr/s:principle | s:tr/s:requirement">
    <td>
      <a name="{@id}" id="{@id}" title="{@name}">
        <xsl:value-of select="@name"/>
      </a>
    </td>
    <td>
      <xsl:apply-templates select="s:short"/>
    </td>
    <td>
      <xsl:apply-templates select="s:long"/>
    </td>
  </xsl:template>

  <xsl:template match="s:principle">
    <span class="principle">
      <a name="{@id}" id="{@id}" title="{@name}"/>
      <span class="principleName"> [Principle <xsl:value-of select="@name"/>] </span>
      <xsl:apply-templates select="s:short"/> - <xsl:apply-templates select="s:long"/>
    </span>
  </xsl:template>

  <xsl:template match="s:requirement">
    <span class="requirement">
      <a name="{@id}" id="{@id}" title="{@name}"/>
      <span class="requirementName"> [Requirement <xsl:value-of select="@name"/>] </span>
      <xsl:apply-templates select="s:short"/> - <xsl:apply-templates select="s:long"/>
    </span>
  </xsl:template>

  <xsl:template match="s:short">
    <span class="short">
      <xsl:apply-templates/>
    </span>
  </xsl:template>

  <xsl:template match="s:short" mode="toc">
    <span>
      <xsl:apply-templates/>
    </span>
  </xsl:template>


  <xsl:template match="s:long">
    <xsl:apply-templates/>
  </xsl:template>

  <xsl:template match="s:error">
    <span class="errordefinition">
      <xsl:call-template name="if-diff"/>
      <a name="{@id}" id="{@id}"/>
      <xsl:apply-templates/>
      <xsl:apply-templates select="@id" mode="conformance"/>
    </span>

    <xsl:if test="count(s:errorcode)!=1">
      <span class="error">
        <xsl:text>This error does not define an error code.</xsl:text>
      </span>
    </xsl:if>

  </xsl:template>
  <xsl:template match="s:error2">
    <span class="errordefinition">
      <xsl:call-template name="if-diff"/>
      <a name="{@id}" id="{@id}"/>
      <xsl:apply-templates/>
      <xsl:apply-templates select="@id" mode="conformance"/>
    </span>

    <xsl:if test="count(s:errorcode2)!=1">
      <span class="error">
        <xsl:text>This error does not define an error code.</xsl:text>
      </span>
    </xsl:if>

  </xsl:template>

  <xsl:template match="s:errorcode">
    <xsl:apply-templates select="key('ids', 'ns-errors')[1]">
      <xsl:with-param name="use" select="'prefix'"/>
    </xsl:apply-templates>
    <xsl:call-template name="colon"/>
    <xsl:value-of select="@code"/>
  </xsl:template>
  <xsl:template match="s:errorcode2">
    <xsl:apply-templates select="key('ids', 'ns-errors2')[1]">
      <xsl:with-param name="use" select="'prefix'"/>
    </xsl:apply-templates>
    <xsl:call-template name="colon"/>
    <xsl:value-of select="@code"/>
  </xsl:template>

  <xsl:template match="s:quote">
    <xsl:call-template name="quote"/>
    <xsl:apply-templates/>
    <xsl:call-template name="quote"/>
  </xsl:template>

  <!-- ============================================================= -->
  <!-- List templates -->
  <!-- ============================================================= -->

  <xsl:template match="s:olist">
    <ol>
      <xsl:call-template name="if-diff"/>
      <xsl:apply-templates/>
    </ol>
  </xsl:template>

  <xsl:template match="s:ulist">
    <ul>
      <xsl:call-template name="if-diff"/>
      <xsl:apply-templates/>
    </ul>
  </xsl:template>

  <xsl:template match="s:slist">
    <ul class="no-bullets">
      <xsl:call-template name="if-diff"/>
      <xsl:apply-templates/>
    </ul>
  </xsl:template>

  <xsl:template match="s:item">
    <li>
      <xsl:call-template name="if-diff"/>
      <xsl:apply-templates/>
    </li>
  </xsl:template>

  <!-- ============================================================= -->
  <!-- Table templates -->
  <!-- ============================================================= -->

  <xsl:template match="s:table">
    <xsl:apply-templates select="s:caption" mode="table.heading"/>
    <table>
      <xsl:for-each select="@*">
        <!-- Wait: some of these aren't HTML attributes after all... -->
        <xsl:choose>
          <xsl:when test="local-name(.) = 'role'">
            <xsl:attribute name="class">
              <xsl:value-of select="."/>
            </xsl:attribute>
          </xsl:when>
          <xsl:when test="local-name(.) = 'id'">
            <!-- nop -->
          </xsl:when>
          <xsl:when test="local-name(.) = 'diff'">
            
            <!-- nop -->
          </xsl:when>
          <xsl:otherwise>
            <xsl:copy-of select="."/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:for-each>  
      <xsl:call-template name="if-diff"/>
      <xsl:apply-templates/>

    </table>
    <xsl:if test=".//s:footnote">
      <p>
        <xsl:apply-templates select=".//s:footnote" mode="table.notes"/>
      </p>
    </xsl:if>
  </xsl:template>

  <xsl:template match="s:caption[parent::s:table]"/>

  <xsl:template match="s:caption" mode="table.heading">
    <div class="chunkHeading">
      <xsl:call-template name="if-diff"/>
      <a name="{../@id}" id="{../@id}"/>
      <xsl:text>Table </xsl:text>
      <xsl:value-of select="count(preceding::s:caption[parent::s:table])+1"/>
      <xsl:text>: </xsl:text>
      <xsl:apply-templates/>
      <xsl:apply-templates select="../@id" mode="conformance"/>
    </div>
  </xsl:template>

  <xsl:template match="s:col|s:colgroup|s:tfoot|s:thead|s:tr|s:tbody">
    <xsl:element name="{local-name(.)}">
      <xsl:for-each select="@*">
        <!-- Wait: some of these aren't HTML attributes after all... -->
        <xsl:choose>
          <xsl:when test="local-name(.) = 'role'">
            <xsl:attribute name="class">
              <xsl:value-of select="."/>
            </xsl:attribute>
          </xsl:when>
          <xsl:when test="local-name(.) = 'diff'">
            
            <!-- nop -->
          </xsl:when>
          <xsl:otherwise>
            <xsl:copy>
              <xsl:apply-templates select="."/>
            </xsl:copy>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:for-each>
      <xsl:call-template name="if-diff"/>
      <xsl:apply-templates/>
    </xsl:element>
  </xsl:template>

  <xsl:template match="s:td|s:th">
    <xsl:element name="{local-name(.)}">
      <xsl:for-each select="@*">
        <xsl:choose>
          <xsl:when test="local-name(.) = 'role'">
            <xsl:attribute name="class">
              <xsl:value-of select="."/>
            </xsl:attribute>
          </xsl:when>
          <xsl:when test="local-name(.) = 'diff'">
            
            <!-- nop -->
          </xsl:when>
          <xsl:when test="local-name(.) = 'colspan' and number(.) = 1"/>
          <xsl:when test="local-name(.) = 'rowspan' and number(.) = 1"/>
          <xsl:otherwise>
            <xsl:copy-of select="."/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:for-each>
      <xsl:call-template name="if-diff"/>
      <xsl:apply-templates/>
    </xsl:element>
  </xsl:template>

  <xsl:template match="s:footnote[ancestor::s:table]">
    <xsl:variable name="this-note-id">
      <xsl:choose>
        <xsl:when test="../@id">
          <xsl:value-of select="../@id"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="generate-id(parent::*)"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <span class="table.footnote">
      <sup>
        <a name="{$this-note-id}" id="{$this-note-id}" href="#FN-ANCH-{$this-note-id}">
          <xsl:apply-templates select="." mode="table-note-number-simple"/>
        </a>
      </sup>
    </span>
  </xsl:template>

  <xsl:template match="s:footnote">
    <xsl:variable name="this-note-id">
      <xsl:choose>
        <xsl:when test="@id">
          <xsl:value-of select="@id"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="generate-id(.)"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <span class="footnote">
      <xsl:call-template name="if-diff"/>
      <sup> [ <a name="{$this-note-id}" id="{$this-note-id}" href="#FN-ANCH-{$this-note-id}">
          <xsl:apply-templates select="." mode="number-simple"/>
        </a> ] </sup>
    </span>
  </xsl:template>

  <xsl:template match="s:footnote" mode="table.notes">
    <xsl:variable name="this-note-id">
      <xsl:choose>
        <xsl:when test="../@id">
          <xsl:value-of select="../@id"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="generate-id(parent::*)"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <span class="table.footnote">
      <xsl:call-template name="if-diff"/>
      <a name="FN-ANCH-{$this-note-id}" id="FN-ANCH-{$this-note-id}" href="#{$this-note-id}">
        <xsl:apply-templates select="." mode="table-note-number-simple"/>
        <xsl:text>.</xsl:text>
      </a>
      <xsl:text> </xsl:text>
      <xsl:apply-templates/>
    </span>
  </xsl:template>

  <xsl:template match="s:footnote" mode="table-note-number-simple">
    <xsl:variable name="table" select="ancestor::s:table"/>
    <xsl:variable name="number" select="count(preceding::s:footnote[ancestor::s:table = $table])+1"/>
    <xsl:value-of select="$number"/>
  </xsl:template>

  <xsl:template match="s:footnote" mode="number-simple">
    <!-- xsl:number level="any" format="1"/-->
    <xsl:variable name="number" select="count(preceding::s:footnote[count(ancestor::s:table) = 0])+1"/>
    <xsl:value-of select="$number"/>
  </xsl:template>

  <!-- ============================================================= -->
  <!-- Boilerplate templates -->
  <!-- ============================================================= -->

  <xsl:template name="banner"> </xsl:template>

  <xsl:template name="status">
    <div>
      <xsl:call-template name="if-diff"/>
      <h2>
        <xsl:call-template name="anchor">
          <xsl:with-param name="conditional" select="0"/>
          <xsl:with-param name="default.id" select="'status'"/>
        </xsl:call-template>
        <xsl:text>Status</xsl:text>
      </h2>

      <p>
        <xsl:variable name="wg" select="/s:spec/s:header/s:wg"/>
        <xsl:variable name="pretext">
          <xsl:choose>
            <xsl:when test="$parentId!=''">
              <xsl:text> supporting document for a </xsl:text>
            </xsl:when>
            <xsl:otherwise/>
          </xsl:choose>
        </xsl:variable>
        <xsl:choose>
          <xsl:when test="$status='WGWD'">
            <xsl:value-of
              select="concat('Circulation of this ',$pretext,' Working Group Working Draft is ',
              'restricted to members and observers of the ',$wg,' of XBRL International and is ',
              'not appropriate to cite from other sources. ')"
            />
          </xsl:when>
          <xsl:when test="$status='IWD'">
            <xsl:value-of
              select="concat('Circulation of this ',$pretext,
            ' Internal Working Draft is restricted to participants in XBRL International ',
            'and is not appropriate to cite from other sources. ')"
            />
          </xsl:when>
          <xsl:when test="$status='DPWD'">
            <xsl:value-of
              select="concat('Circulation of this ',$pretext,
            ' Draft Public Working Draft is restricted to participants in XBRL International ',
            'and is not appropriate to cite from other sources. ')"
            />
          </xsl:when>
          <xsl:when test="$status='PWD'">
            <xsl:value-of
              select="concat('Circulation of this ',$pretext,
            ' Public Working Draft is unrestricted. ')"/>
          </xsl:when>
          <xsl:when test="$status='DWGN'">
            <xsl:value-of
              select="concat('Circulation of this ',$pretext,
            ' Draft Working Group Note is restricted to participants in XBRL International ',
            'and is not appropriate to cite from other sources. ')"
            />
          </xsl:when>
          <xsl:when test="$status='WGN'">
            <xsl:text>
              Circulation of this Working Group Note is unrestricted. 
            </xsl:text>
          </xsl:when>
          <xsl:when test="$status='DFWGN'">
            <xsl:value-of
              select="concat('Circulation of this ',$pretext,
            ' Draft Final Working Group Note is restricted to participants in XBRL International ',
            'and is not appropriate to cite from other sources. ')"
            />
          </xsl:when>
          <xsl:when test="$status='FWGN'">
            <xsl:text>
              Circulation of this Final Working Group Note is unrestricted. 
            </xsl:text>
          </xsl:when>
          <xsl:when test="$status='DCR'">
            <xsl:value-of
              select="concat('Circulation of this ',$pretext,
            ' Draft Candidate Recommendation is restricted to participants in XBRL International ',
            'and is not appropriate to cite from other sources. ')"
            />
          </xsl:when>
          <xsl:when test="$status='DR'">
            <xsl:value-of
              select="concat('Circulation of this ',$pretext,
            ' Draft Recommendation is restricted to participants in XBRL International ',
            'and is not appropriate to cite from other sources. ')"
            />
          </xsl:when>
          <xsl:when test="$status='CR'">
            <xsl:value-of
              select="concat('Circulation of this ',$pretext,
            ' Candidate Recommendation is unrestricted. ')"
            />
          </xsl:when>
          <xsl:when test="$status='DPR'">
            <xsl:value-of
              select="concat('Circulation of this ',$pretext,
            ' Draft Proposed Recommendation is restricted to participants in XBRL International ',
            'and is not appropriate to cite from other sources. ')"
            />
          </xsl:when>
          <xsl:when test="$status='PR'">
            <xsl:value-of
              select="concat('Circulation of this ',$pretext,
            ' Proposed Recommendation is unrestricted. ')"
            />
          </xsl:when>
          <xsl:when test="$status='REC'">
            <xsl:value-of
              select="concat('Circulation of this ',$pretext,
            ' Recommendation is unrestricted. ')"/>
          </xsl:when>
          <xsl:when test="$status='DPER'">
            <xsl:value-of
              select="concat('Circulation of this ',$pretext,
            ' Draft Proposed Edited Recommendation is restricted to participants in XBRL International ',
            'and is not appropriate to cite from other sources. ')"
            />
          </xsl:when>
          <xsl:when test="$status='PER'">
            <xsl:value-of
              select="concat('Circulation of this ',$pretext,
            ' Proposed Edited Recommendation is unrestricted. ')"
            />
          </xsl:when>
          <xsl:when test="$status='DRR'">
            <xsl:value-of
              select="concat('Circulation of this ',$pretext,
            ' Draft Rescinded Recommendation is restricted to participants in XBRL International ',
            'and is not appropriate to cite from other sources. ')"
            />
          </xsl:when>
          <xsl:when test="$status='PER'">
            <xsl:value-of
              select="concat('Circulation of this ',$pretext,
            ' Rescinded Recommendation is unrestricted. ')"
            />
          </xsl:when>
        </xsl:choose>
        <xsl:choose>
          <xsl:when test="$pubType!='Specification'">
            <xsl:text/>
          </xsl:when>
          <xsl:when test="$parentId!=''">
            <xsl:text/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:text>This document is normative. </xsl:text>
          </xsl:otherwise>
        </xsl:choose>
        <xsl:if test="$status!='REC'">
          <xsl:text>
            Other documents may supersede this document. 
          </xsl:text>
        </xsl:if>
        <xsl:text>
          Recipients are invited to submit comments to
        </xsl:text>
        <xsl:apply-templates select="/s:spec/s:header/s:feedback"/>,
        <xsl:text>
          and to submit notification of any relevant patent rights 
          of which they are aware and provide supporting 
          documentation.
        </xsl:text>
      </p>

    </div>
  </xsl:template>

  <xsl:template match="s:feedback">
    <a href="mailto:{.}">
      <xsl:call-template name="if-diff"/>
      <xsl:value-of select="."/>
    </a>
  </xsl:template>

  <xsl:template name="endnotes">
    <xsl:if test="//s:footnote[not(ancestor::s:table)]">
      <hr/>
      <div class="endnotes">
        <xsl:call-template name="newline"/>
        <h3>
          <xsl:call-template name="anchor">
            <xsl:with-param name="conditional" select="0"/>
            <xsl:with-param name="default.id" select="'endnotes'"/>
          </xsl:call-template>
          <xsl:text>End Notes</xsl:text>
        </h3>
        <dl>
          <xsl:apply-templates select="//s:footnote[not(ancestor::s:table)]" mode="endnotes"/>
        </dl>
      </div>
    </xsl:if>
  </xsl:template>

  <xsl:template match="s:footnote" mode="endnotes">
    <xsl:variable name="this-note-id">
      <xsl:choose>
        <xsl:when test="@id">
          <xsl:value-of select="@id"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="generate-id(.)"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <dt>
      <xsl:call-template name="if-diff"/>
      <xsl:text>[</xsl:text>
      <a name="FN-ANCH-{$this-note-id}" id="FN-ANCH-{$this-note-id}" href="#{$this-note-id}">
        <xsl:apply-templates select="." mode="number-simple"/>
      </a>
      <xsl:text>]</xsl:text>
    </dt>
    <dd>
      <xsl:call-template name="if-diff"/>
      <xsl:apply-templates/>
    </dd>
  </xsl:template>

  <!-- ============================================================= -->
  <!-- Bibliography -->
  <!-- ============================================================= -->

  <xsl:template match="s:bibliography">
    <dl>
      <xsl:call-template name="if-diff"/>
      <xsl:apply-templates>
        <xsl:sort select="@key"/>
      </xsl:apply-templates>
    </dl>
  </xsl:template>

  <xsl:template match="s:reference">
    <dt class="label">
      <xsl:call-template name="if-diff"/>
      <xsl:if test="@id">
        <a name="{@id}" id="{@id}"/>
      </xsl:if>
      <xsl:choose>
        <xsl:when test="@key">
          <xsl:value-of select="@key"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="@id"/>
        </xsl:otherwise>
      </xsl:choose>
      <xsl:variable name="id" select="@id"/>
      <xsl:if test="count(//s:bibref[@ref=$id])+ count(//s:xtermref[@bibref=$id])=0">
        <span class="error">This reference is not cited.</span>
      </xsl:if>
    </dt>
    <dd>
      <xsl:call-template name="if-diff"/>
      <xsl:apply-templates select="s:publisher" mode="reference"/>
      <xsl:apply-templates select="s:title" mode="reference"/>
      <xsl:apply-templates select="s:authors/s:person" mode="reference"/>
      <xsl:if test="@href">
        <br/>
        <xsl:text>(See </xsl:text>
        <a href="{@href}">
          <xsl:value-of select="@href"/>
        </a>
        <xsl:text>)</xsl:text>
      </xsl:if>
    </dd>
  </xsl:template>

  <xsl:template match="s:title" mode="reference">
    <xsl:text> </xsl:text>
    <xsl:call-template name="quote"/>
    <xsl:apply-templates/>
    <xsl:call-template name="quote"/>
  </xsl:template>

  <xsl:template match="s:publisher" mode="reference">
    <xsl:apply-templates/>
    <xsl:text>.</xsl:text>
  </xsl:template>

  <xsl:template match="s:person" mode="reference">
    <span>
      <xsl:call-template name="if-diff"/>
      <xsl:if test="count(preceding-sibling::s:person)=0">
        <br/>
      </xsl:if>
      <xsl:if test="preceding-sibling::s:person ">
        <xsl:text>, </xsl:text>
      </xsl:if>
      <xsl:if test="count(../s:person)&gt;1 and count(following-sibling::s:person)=0">
        <xsl:text>and </xsl:text>
      </xsl:if>
      <xsl:apply-templates select="s:name"/>
      <xsl:if test="count(following-sibling::s:person)=0">
        <xsl:text>.</xsl:text>
      </xsl:if>
    </span>
  </xsl:template>

  <!-- ============================================================= -->
  <!-- Revisions -->
  <!-- ============================================================= -->

  <xsl:template match="s:revisions">
    <table class="revisions">
      <tr>
        <th>Date</th>
        <th>Author</th>
        <th>Details</th>
      </tr>
      <xsl:apply-templates/>
    </table>
  </xsl:template>

  <xsl:template match="s:revision">
    <tr>
      <td>
        <xsl:value-of select="@day"/>
        <xsl:call-template name="space"/>
        <xsl:call-template name="month">
          <xsl:with-param name="month" select="@month"/>
        </xsl:call-template>
        <xsl:call-template name="space"/>
        <xsl:value-of select="@year"/>
      </td>
      <td>
        <xsl:apply-templates select="key('ids', @ref)/s:name"/>
      </td>
      <td>
        <xsl:apply-templates/>
      </td>
    </tr>
  </xsl:template>

  <!-- ============================================================= -->
  <!-- Errata -->
  <!-- ============================================================= -->

  <xsl:template match="s:errata">
    <p>No errata have been incorporated into this document.</p>
  </xsl:template>

  <xsl:template match="s:errata[s:erratum]">
    <table class="errata">
      <tr>
        <th>Number</th>
        <th>Date</th>
        <th>Sections</th>
        <th>Details</th>
      </tr>
      <xsl:apply-templates/>
    </table>
  </xsl:template>

  <xsl:template match="s:erratum">
    <tr>
      <xsl:call-template name="if-diff"/>
      <td>
        <xsl:value-of select="count(preceding-sibling::s:erratum)+1"/>
        <xsl:text>.</xsl:text>
      </td>
      <td>
        <xsl:value-of select="@day"/>
        <xsl:call-template name="space"/>
        <xsl:call-template name="month">
          <xsl:with-param name="month" select="@month"/>
        </xsl:call-template>
        <xsl:call-template name="space"/>
        <xsl:value-of select="@year"/>
      </td>
      <td>
        <xsl:for-each select="s:sectionref">
          <xsl:if test="count(preceding-sibling::s:sectionref)">
            <br/>
          </xsl:if>
          <xsl:apply-templates select="."/>
        </xsl:for-each>
      </td>
      <td>
        <xsl:apply-templates select="*[local-name()!='sectionref']"/>
      </td>
    </tr>
  </xsl:template>

  <!-- ============================================================= -->
  <!-- References/ hyperlinks -->
  <!-- ============================================================= -->

  <xsl:template match="s:xref">
    <a href="{@ref}">
      <xsl:choose>
        <xsl:when test="count(child::node())=0">
          <xsl:value-of select="@ref"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:apply-templates/>
        </xsl:otherwise>
      </xsl:choose>
    </a>
  </xsl:template>

  <xsl:template match="s:xtermref">
    <xsl:variable name="reference" select="key('ids', @bibref)"/>
    <a>
      <xsl:attribute name="href">
        <xsl:value-of select="$reference/@href"/>
        <xsl:if test="count(@idref)">
          <xsl:text>#</xsl:text>
          <xsl:value-of select="normalize-space(@idref)"/>
        </xsl:if>
      </xsl:attribute>
      <xsl:choose>
        <xsl:when test="count($reference)=0">
          <span class="error"> No bibliographic citation has ID <xsl:value-of select="@bibref"/> . </span>
        </xsl:when>
        <xsl:when test="count(child::node())=0">
          <span class="error"> No term is linked to the specified external term reference. </span>
        </xsl:when>
        <xsl:otherwise>
          <xsl:apply-templates/>
        </xsl:otherwise>
      </xsl:choose>
    </a>
  </xsl:template>

  <xsl:template match="s:termref">
    <xsl:param name="target" select="key('ids', @ref)[1]"/>
    <xsl:choose>
      <xsl:when test="$target">
        <xsl:apply-templates select="$target" mode="termref">
          <xsl:with-param name="usage" select="node()"/>
        </xsl:apply-templates>
      </xsl:when>
      <xsl:otherwise>
        <span class="error"> [Cannot resolve Definition ID <xsl:value-of select="@ref"/> ] </span>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="s:definition" mode="termref">
    <xsl:param name="usage" select="/.."/>
    <a>
      <xsl:attribute name="href">
        <xsl:call-template name="href.target"/>
      </xsl:attribute>
      <xsl:call-template name="if-diff"/>
      <xsl:apply-templates select="$usage"/>
    </a>
  </xsl:template>

  <xsl:template match="s:principleref">
    <xsl:param name="target" select="key('ids', @ref)[1]"/>
    <xsl:choose>
      <xsl:when test="$target">
        <xsl:apply-templates select="$target" mode="principleref"/>
      </xsl:when>
      <xsl:otherwise>
        <span class="error"> [Cannot resolve Principle ID <xsl:value-of select="@ref"/> ] </span>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="s:principle" mode="principleref">
    <a>
      <xsl:attribute name="href">
        <xsl:call-template name="href.target"/>
      </xsl:attribute> Principle <xsl:value-of select="@name"/>
      <xsl:call-template name="if-diff"/>
    </a>
  </xsl:template>

  <xsl:template match="s:requirementref">
    <xsl:param name="target" select="key('ids', @ref)[1]"/>
    <xsl:choose>
      <xsl:when test="$target">
        <xsl:apply-templates select="$target" mode="requirementref"/>
      </xsl:when>
      <xsl:otherwise>
        <span class="error"> [Cannot resolve Requirement ID <xsl:value-of select="@ref"/>] </span>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="s:requirement" mode="requirementref">
    <a>
      <xsl:attribute name="href">
        <xsl:call-template name="href.target"/>
      </xsl:attribute> Requirement <xsl:value-of select="@name"/>
    </a>
  </xsl:template>

  <xsl:template match="s:sectionref">
    <xsl:param name="target" select="key('ids', @ref)[1]"/>
    <xsl:choose>
      <xsl:when test="$target">
        <xsl:apply-templates select="$target" mode="sectionref"/>
      </xsl:when>
      <xsl:otherwise>
        <span class="error"> [Cannot resolve Section ID <xsl:value-of select="@ref"/>] </span>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="s:section" mode="sectionref">
    <a>
      <xsl:attribute name="href">
        <xsl:call-template name="href.target"/>
      </xsl:attribute>
      <b>
        <xsl:choose>
          <xsl:when test="ancestor::s:body"> Section <xsl:apply-templates select="." mode="divnum"/>
          </xsl:when>
          <xsl:when test="ancestor::s:back">
            <xsl:apply-templates select="." mode="divnum"/>
          </xsl:when>
        </xsl:choose>
      </b>
    </a>
  </xsl:template>

  <xsl:template match="s:bibref">
    <a>
      <xsl:attribute name="href">
        <xsl:call-template name="href.target">
          <xsl:with-param name="target" select="key('ids', @ref)"/>
        </xsl:call-template>
      </xsl:attribute>
      <xsl:text>[</xsl:text>
      <xsl:value-of select="key('ids', @ref)/@key"/>
      <xsl:text>]</xsl:text>
    </a>
  </xsl:template>

  <xsl:template match="s:figureref">
    <xsl:param name="target" select="key('ids', @ref)[1]"/>
    <xsl:choose>
      <xsl:when test="$target">
        <xsl:apply-templates select="$target" mode="figureref"/>
      </xsl:when>
      <xsl:otherwise>
        <span class="error"> [Cannot resolve Figure ID <xsl:value-of select="@ref"/> ] </span>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="s:figure" mode="figureref">
    <a>
      <xsl:attribute name="href">
        <xsl:call-template name="href.target"/>
      </xsl:attribute>
      <b> Figure <xsl:value-of select="count(preceding::s:caption[parent::s:figure])+1"/>
      </b>
    </a>
  </xsl:template>

  <xsl:template match="s:exampleref">
    <xsl:param name="target" select="key('ids', @ref)[1]"/>
    <xsl:choose>
      <xsl:when test="$target">
        <xsl:apply-templates select="$target" mode="exampleref"/>
      </xsl:when>
      <xsl:otherwise>
        <span class="error"> [Cannot resolve Example ID <xsl:value-of select="@ref"/> ] </span>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="s:example" mode="exampleref">
    <a>
      <xsl:attribute name="href">
        <xsl:call-template name="href.target"/>
      </xsl:attribute>
      <b> Example <xsl:value-of select="count(preceding::s:caption[parent::s:example])+1"/>
      </b>
    </a>
  </xsl:template>

  <xsl:template match="s:tableref">
    <xsl:param name="target" select="key('ids', @ref)[1]"/>
    <xsl:choose>
      <xsl:when test="$target">
        <xsl:apply-templates select="$target" mode="tableref"/>
      </xsl:when>
      <xsl:otherwise>
        <span class="error"> [Cannot resolve Table ID <xsl:value-of select="@ref"/> ] </span>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="s:table" mode="tableref">
    <a>
      <xsl:attribute name="href">
        <xsl:call-template name="href.target"/>
      </xsl:attribute>
      <b> Table <xsl:value-of select="count(preceding::s:caption[parent::s:table])+1"/>
      </b>
    </a>
  </xsl:template>

  <xsl:template match="s:personref">
    <xsl:param name="target" select="key('ids', @ref)[1]"/>
    <xsl:choose>
      <xsl:when test="$target">
        <xsl:apply-templates select="$target" mode="personref"/>
      </xsl:when>
      <xsl:otherwise>
        <span class="error"> [Cannot resolve Person ID <xsl:value-of select="@ref"/> ] </span>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="s:person" mode="personref">
    <a>
      <xsl:attribute name="href">
        <xsl:call-template name="href.target"/>
      </xsl:attribute>
      <b>
        <xsl:apply-templates select="s:name"/>
      </b>
    </a>
  </xsl:template>

  <xsl:template match="s:namespaceref">
    <xsl:param name="target" select="key('ids', @ref)[1]"/>
    <xsl:choose>
      <xsl:when test="$target">
        <xsl:apply-templates select="$target">
          <xsl:with-param name="use" select="@use"/>
        </xsl:apply-templates>
      </xsl:when>
      <xsl:otherwise>
        <span class="error"> [Cannot resolve Person ID <xsl:value-of select="@ref"/> ] </span>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="s:namespace">
    <xsl:param name="use" select="'prefix'"/>
    <xsl:choose>
      <xsl:when test="$use='namespace'">
        <xsl:value-of select="@value"/>
      </xsl:when>
      <xsl:when test="$use='url' and @url">
        <a href="{@url}">
          <xsl:value-of select="@url"/>
        </a>
      </xsl:when>
      <xsl:when test="$use='url'">
        <span class="error"> [Namespace <xsl:value-of select="@value"/> has no normative schema URL.] </span>
      </xsl:when>
      <xsl:when test="$use='prefix'">
        <xsl:value-of select="@prefix"/>
      </xsl:when>
      <xsl:when test="$use='description'">
        <xsl:value-of select="text()"/>
      </xsl:when>
      <xsl:otherwise>
        <span class="error"> [ <xsl:value-of select="$use"/> is an invalid use of namespace <xsl:value-of
            select="@value"/> ] </span>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="s:xmlref">
    <xsl:param name="target" select="key('ids', @ref)[1]"/>
    <xsl:choose>
      <xsl:when test="$target">
        <xsl:apply-templates select="$target" mode="xmlref">
          <xsl:with-param name="usage" select="node()"/>
        </xsl:apply-templates>
      </xsl:when>
      <xsl:otherwise>
        <span class="error"> [Cannot resolve XML ID <xsl:value-of select="@ref"/> ] </span>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="s:errorref">
    <xsl:param name="target" select="key('ids', @ref)[1]"/>
    <xsl:choose>
      <xsl:when test="$target">
        <xsl:apply-templates select="$target" mode="errorref">
          <xsl:with-param name="usage" select="node()"/>
        </xsl:apply-templates>
      </xsl:when>
      <xsl:otherwise>
        <span class="error"> [Cannot resolve error ID <xsl:value-of select="@ref"/> ] </span>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>
  <xsl:template match="s:errorref2">
    <xsl:param name="target" select="key('ids', @ref)[1]"/>
    <xsl:choose>
      <xsl:when test="$target">
        <xsl:apply-templates select="$target" mode="errorref">
          <xsl:with-param name="usage" select="node()"/>
        </xsl:apply-templates>
      </xsl:when>
      <xsl:otherwise>
        <span class="error"> [Cannot resolve error ID <xsl:value-of select="@ref"/> ] </span>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="s:error" mode="errorref">
    <xsl:param name="usage" select="/.."/>
    <a>
      <xsl:attribute name="href">
        <xsl:call-template name="href.target"/>
      </xsl:attribute>
      <xsl:apply-templates select="$usage"/>
      <xsl:apply-templates select="s:errorcode"/>
    </a>
  </xsl:template>
  <xsl:template match="s:error2" mode="errorref">
    <xsl:param name="usage" select="/.."/>
    <a>
      <xsl:attribute name="href">
        <xsl:call-template name="href.target"/>
      </xsl:attribute>
      <xsl:apply-templates select="$usage"/>
      <xsl:apply-templates select="s:errorcode2"/>
    </a>
  </xsl:template>

  <xsl:template match="*" mode="xmlref">
    <xsl:param name="usage" select="/.."/>
    <a>
      <xsl:attribute name="href">
        <xsl:call-template name="href.target"/>
      </xsl:attribute>
      <xsl:apply-templates select="$usage"/>
    </a>
  </xsl:template>

  <!-- ============================================================= -->
  <!-- Templates used as functions -->
  <!-- ============================================================= -->

  <xsl:template name="anchor">
    <xsl:param name="node" select="."/>
    <xsl:param name="conditional" select="1"/>
    <xsl:param name="default.id" select="''"/>
    <xsl:variable name="id">
      <xsl:call-template name="element.id">
        <xsl:with-param name="node" select="$node"/>
        <xsl:with-param name="default.id" select="$default.id"/>
      </xsl:call-template>
    </xsl:variable>
    <xsl:if test="$conditional=0 or $node/@id">
      <a name="{$id}" id="{$id}"/>
    </xsl:if>
  </xsl:template>

  <xsl:template name="element.id">
    <xsl:param name="node" select="."/>
    <xsl:param name="default.id" select="''"/>
    <xsl:choose>
      <xsl:when test="$default.id != '' and not(key('ids', $default.id))">
        <xsl:value-of select="$default.id"/>
      </xsl:when>
      <xsl:when test="$node/@id">
        <xsl:value-of select="$node/@id"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="generate-id($node)"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template name="add-id">
    <xsl:param name="node" select="."/>
    <xsl:param name="default.id" select="''"/>
    <xsl:variable name="id">
      <xsl:call-template name="element.id">
        <xsl:with-param name="node" select="$node"/>
        <xsl:with-param name="default.id" select="$default.id"/>
      </xsl:call-template>
    </xsl:variable>
    <xsl:if test="@id">
      <xsl:attribute name="id">
        <xsl:value-of select="$id"/>
      </xsl:attribute>
    </xsl:if>
  </xsl:template>

  <xsl:template name="status-text">
    <xsl:choose>
      <xsl:when test="$parentId!=''">
        <xsl:text>Supporting document for a </xsl:text>
      </xsl:when>
    </xsl:choose>

    <xsl:choose>
      <xsl:when test="$status='WGN'">
        <xsl:text>Working Group Note</xsl:text>
      </xsl:when>
      <xsl:when test="$status='IWD'">
        <xsl:text>Internal Working Draft</xsl:text>
      </xsl:when>
      <xsl:when test="$status='PWD'">
        <xsl:text>Public Working Draft</xsl:text>
      </xsl:when>
      <xsl:when test="$status='CR'">
        <xsl:text>Candidate Recommendation</xsl:text>
      </xsl:when>
      <xsl:when test="$status='REC'">
        <xsl:text>Recommendation</xsl:text>
      </xsl:when>
      <xsl:when test="$status='RR'">
        <xsl:text>Rescinded Recommendation</xsl:text>
      </xsl:when>
      <xsl:when test="$status='DRR'">
        <xsl:text>Draft Rescinded Recommendation</xsl:text>
      </xsl:when>
      <xsl:when test="$status='PER'">
        <xsl:text>Proposed Edited Recommendation</xsl:text>
      </xsl:when>
      <xsl:when test="$status='DPER'">
        <xsl:text>Draft Proposed Edited Recommendation</xsl:text>
      </xsl:when>
      <xsl:when test="$status='PR'">
        <xsl:text>Proposed Recommendation</xsl:text>
      </xsl:when>
      <xsl:when test="$status='DPR'">
        <xsl:text>Draft Proposed Recommendation</xsl:text>
      </xsl:when>
      <xsl:when test="$status='DCR'">
        <xsl:text>Draft Candidate Recommendation</xsl:text>
      </xsl:when>
      <xsl:when test="$status='DR'">
        <xsl:text>Draft Recommendation</xsl:text>
      </xsl:when>
      <xsl:when test="$status='FWGN'">
        <xsl:text>Final Working Group Note</xsl:text>
      </xsl:when>
      <xsl:when test="$status='DFWGN'">
        <xsl:text>Draft Final Working Group Note</xsl:text>
      </xsl:when>
      <xsl:when test="$status='DWGN'">
        <xsl:text>Draft Working Group Note</xsl:text>
      </xsl:when>
      <xsl:when test="$status='DPWD'">
        <xsl:text>Draft Public Working Draft</xsl:text>
      </xsl:when>
      <xsl:when test="$status='WGWD'">
        <xsl:text>Working Group Working Draft</xsl:text>
      </xsl:when>
    </xsl:choose>
  </xsl:template>

  <xsl:template name="a-status">

    <xsl:choose>
      <xsl:when test="$status='IWD'">
        <xsl:text>an </xsl:text>
      </xsl:when>
      <xsl:otherwise>
        <xsl:text>a </xsl:text>
      </xsl:otherwise>
    </xsl:choose>
    <xsl:call-template name="status-text"/>
  </xsl:template>

  <xsl:template match="s:dateref">

    <xsl:choose>
      <xsl:when test="@style='DD-MM-YYYY'">
        <xsl:value-of select="concat($date/@day,'/',$date/@month,'/',$date/@year)"/>
      </xsl:when>
      <xsl:when test="@style='YYYYMMDD'">
        <xsl:value-of select="concat($date/@year,$date/@month,$date/@day)"/>
      </xsl:when>
      <xsl:when test="@style='DD Month YYYY'">
        <xsl:call-template name="full-date"/>
      </xsl:when>
      <xsl:otherwise>
        <span class="error"> [Invalid date style=' <xsl:value-of select="@style"/> '] </span>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template name="full-date">

    <xsl:if test="$date/@day">
      <xsl:apply-templates select="$date/@day"/>
      <xsl:call-template name="space"/>
    </xsl:if>
    <xsl:call-template name="month"/>
    <xsl:call-template name="space"/>
    <xsl:apply-templates select="$date/@year"/>
  </xsl:template>

  <xsl:template name="full-erratarecdate">

    <xsl:if test="$erratarecdate/@day">
      <xsl:apply-templates select="$erratarecdate/@day"/>
      <xsl:call-template name="space"/>
    </xsl:if>
    <xsl:call-template name="erratarecmonth"/>
    <xsl:call-template name="space"/>
    <xsl:apply-templates select="$erratarecdate/@year"/>
  </xsl:template>

  <xsl:template name="month">
    <xsl:param name="month" select="$date/@month"/>
    <xsl:choose>
      <xsl:when test="$month='01'">
        <xsl:text>January</xsl:text>
      </xsl:when>
      <xsl:when test="$month='02'">
        <xsl:text>February</xsl:text>
      </xsl:when>
      <xsl:when test="$month='03'">
        <xsl:text>March</xsl:text>
      </xsl:when>
      <xsl:when test="$month='04'">
        <xsl:text>April</xsl:text>
      </xsl:when>
      <xsl:when test="$month='05'">
        <xsl:text>May</xsl:text>
      </xsl:when>
      <xsl:when test="$month='06'">
        <xsl:text>June</xsl:text>
      </xsl:when>
      <xsl:when test="$month='07'">
        <xsl:text>July</xsl:text>
      </xsl:when>
      <xsl:when test="$month='08'">
        <xsl:text>August</xsl:text>
      </xsl:when>
      <xsl:when test="$month='09'">
        <xsl:text>September</xsl:text>
      </xsl:when>
      <xsl:when test="$month='10'">
        <xsl:text>October</xsl:text>
      </xsl:when>
      <xsl:when test="$month='11'">
        <xsl:text>November</xsl:text>
      </xsl:when>
      <xsl:when test="$month='12'">
        <xsl:text>December</xsl:text>
      </xsl:when>
    </xsl:choose>

  </xsl:template>
  <xsl:template name="erratarecmonth">
    <xsl:param name="month" select="$erratarecdate/@month"/>
    <xsl:choose>
      <xsl:when test="$month='01'">
        <xsl:text>January</xsl:text>
      </xsl:when>
      <xsl:when test="$month='02'">
        <xsl:text>February</xsl:text>
      </xsl:when>
      <xsl:when test="$month='03'">
        <xsl:text>March</xsl:text>
      </xsl:when>
      <xsl:when test="$month='04'">
        <xsl:text>April</xsl:text>
      </xsl:when>
      <xsl:when test="$month='05'">
        <xsl:text>May</xsl:text>
      </xsl:when>
      <xsl:when test="$month='06'">
        <xsl:text>June</xsl:text>
      </xsl:when>
      <xsl:when test="$month='07'">
        <xsl:text>July</xsl:text>
      </xsl:when>
      <xsl:when test="$month='08'">
        <xsl:text>August</xsl:text>
      </xsl:when>
      <xsl:when test="$month='09'">
        <xsl:text>September</xsl:text>
      </xsl:when>
      <xsl:when test="$month='10'">
        <xsl:text>October</xsl:text>
      </xsl:when>
      <xsl:when test="$month='11'">
        <xsl:text>November</xsl:text>
      </xsl:when>
      <xsl:when test="$month='12'">
        <xsl:text>December</xsl:text>
      </xsl:when>
    </xsl:choose>

  </xsl:template>

  <xsl:template match="s:wgref">
    <xsl:value-of select="/s:spec/s:header/s:wg"/>
  </xsl:template>

  <xsl:template name="newline">
    <xsl:text>&#10;</xsl:text>
  </xsl:template>

  <xsl:template name="space">
    <xsl:text> </xsl:text>
    <!-- xsl:text>&#160;</xsl:text-->
  </xsl:template>

  <xsl:template name="nbsp">
    <xsl:text>&#160;</xsl:text>
  </xsl:template>


  <xsl:template name="quote">
    <xsl:text>&#34;</xsl:text>
  </xsl:template>

  <xsl:template name="lt">
    <xsl:text>&lt;</xsl:text>
  </xsl:template>

  <xsl:template name="gt">
    <xsl:text>&gt;</xsl:text>
  </xsl:template>

  <xsl:template name="colon">
    <xsl:text>:</xsl:text>
  </xsl:template>

  <xsl:template name="equals">
    <xsl:text>=</xsl:text>
  </xsl:template>

  <xsl:template name="slash">
    <xsl:text>/</xsl:text>
  </xsl:template>

  <xsl:template name="open-pi">
    <xsl:text>&lt;?</xsl:text>
  </xsl:template>
  <xsl:template name="close-pi">
    <xsl:text>?&gt;</xsl:text>
  </xsl:template>

  <xsl:template name="open-comment">
    <xsl:text>&lt;!--</xsl:text>
  </xsl:template>
  <xsl:template name="close-comment">
    <xsl:text>--&gt;</xsl:text>
  </xsl:template>

  <!-- ============================================================= -->
  <!-- Templates used for verbatim XML rendering -->
  <!-- ============================================================= -->

  <xsl:template match="*" mode="verbatim">

    <div class="verbatim">

      <xsl:if test="local-name(../..)='xml' and namespace-uri(../..)=$namespace-uri">
        <div class="verbatim">
          <xsl:call-template name="nbsp"/>
        </div>
      </xsl:if>

      <!-- Left angle bracket -->
      <xsl:call-template name="lt"/>

      <!-- Element namespace prefix -->
      <xsl:variable name="ns-prefix" select="substring-before(name(),':')"/>
      <xsl:if test="$ns-prefix != ''">
        <span class="verbatim-element-nsprefix">
          <xsl:value-of select="$ns-prefix"/>
        </span>
        <xsl:call-template name="colon"/>
      </xsl:if>

      <!-- Element local name -->
      <span class="verbatim-element-name">
        <xsl:value-of select="local-name()"/>
      </span>

      <!-- Namespace prefix declarations -->
      <xsl:variable name="pns" select="../namespace::*"/>
      <!-- Get namespace nodes on parent element -->
      <xsl:if test="$pns[name()=''] and not(namespace::*[name()=''])">
        <!-- Undefine the default namespace if needed -->
        <xsl:call-template name="space"/>
        <span class="verbatim">
          <span class="verbatim-namespace-name">
            <br/>
            <xsl:text> xmlns</xsl:text>
          </span>
          <xsl:call-template name="equals"/>
          <xsl:call-template name="quote"/>
          <xsl:call-template name="quote"/>
        </span>
      </xsl:if>
      <xsl:for-each select="namespace::*">
        <!-- For each namespace on the element -->
        <xsl:choose>
          <xsl:when test="../@xsi:*" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">
            <!-- Generate xsi namespace declarations as needed -->
            <xsl:call-template name="verbatim-namespace"/>
          </xsl:when>
          <xsl:when test="../@s:*">
            <xsl:call-template name="verbatim-namespace"/>
          </xsl:when>
          <xsl:when test="not($pns[name()=name(current()) and .=current()])">
            <xsl:call-template name="verbatim-namespace"/>
          </xsl:when>
        </xsl:choose>
      </xsl:for-each>

      <xsl:apply-templates select="@*" mode="verbatim"/>

      <xsl:choose>
        <xsl:when test="node()">
          <span class="verbatim">
            <xsl:call-template name="gt"/>
          </span>
          <xsl:apply-templates mode="verbatim"/>
          <span class="verbatim">
            <xsl:if test="local-name(..)='xml' and namespace-uri(..)=$namespace-uri">
              <br/>
            </xsl:if>
            <xsl:call-template name="lt"/>
            <xsl:call-template name="slash"/>
            <xsl:if test="$ns-prefix != ''">
              <span class="verbatim-element-nsprefix">
                <xsl:value-of select="$ns-prefix"/>
              </span>
              <xsl:call-template name="colon"/>
            </xsl:if>
            <span class="verbatim-element-name">
              <xsl:value-of select="local-name()"/>
            </span>
            <xsl:call-template name="gt"/>
          </span>
        </xsl:when>
        <xsl:otherwise>
          <span class="verbatim">
            <xsl:call-template name="slash"/>
            <xsl:call-template name="gt"/>
          </span>
        </xsl:otherwise>
      </xsl:choose>
    </div>
  </xsl:template>

  <xsl:template name="verbatim-namespace">
    <xsl:call-template name="space"/>
    <span class="verbatim">
      <xsl:if test="name()!='xml'">
        <span class="verbatim-namespace-name">
          <br/>
          <xsl:text> xmlns</xsl:text>
          <xsl:if test="name()!=''">
            <xsl:call-template name="colon"/>
          </xsl:if>
          <xsl:value-of select="name()"/>
        </span>
        <xsl:call-template name="equals"/>
        <xsl:call-template name="quote"/>
        <span class="verbatim-namespace-uri">
          <xsl:value-of select="."/>
        </span>
        <xsl:call-template name="quote"/>
      </xsl:if>
    </span>
  </xsl:template>

  <xsl:template match="@*" mode="verbatim">
    <xsl:call-template name="space"/>
    <span class="verbatim">
      <span class="verbatim-attribute-name">
        <xsl:if test="local-name()='id'">
          <a name="{.}" id="{.}"/>
          <xsl:apply-templates select="." mode="conformance"/>
        </xsl:if>
        <xsl:value-of select="name()"/>
      </span>
      <xsl:call-template name="equals"/>
      <xsl:call-template name="quote"/>
      <span class="verbatim-attribute-content">
        <xsl:call-template name="html-replace-entities">
          <xsl:with-param name="text" select="normalize-space(.)"/>
          <xsl:with-param name="attrs" select="true()"/>
        </xsl:call-template>
      </span>
      <xsl:call-template name="quote"/>
    </span>
  </xsl:template>

  <xsl:template match="text()" mode="verbatim">
    <xsl:if test="not(normalize-space(.)=' ' or normalize-space(.)='')">
      <div class="verbatim">
        <span class="verbatim-text">
          <xsl:call-template name="preformatted-output">
            <xsl:with-param name="text">
              <xsl:call-template name="html-replace-entities">
                <xsl:with-param name="text" select="."/>
              </xsl:call-template>
            </xsl:with-param>
          </xsl:call-template>
        </span>
      </div>
    </xsl:if>
  </xsl:template>

  <!-- comments -->
  <xsl:template match="comment()" mode="verbatim">
    <div class="verbatim">
      <xsl:call-template name="open-comment"/>
      <span class="verbatim-comment">
        <xsl:call-template name="preformatted-output">
          <xsl:with-param name="text" select="."/>
        </xsl:call-template>
      </span>
      <xsl:call-template name="close-comment"/>
    </div>
  </xsl:template>

  <!-- processing instructions -->
  <xsl:template match="processing-instruction()" mode="verbatim">
    <div class="verbatim">
      <xsl:call-template name="open-pi"/>
      <span class="verbatim-pi-name">
        <xsl:value-of select="name()"/>
      </span>
      <xsl:if test=".!=''">
        <xsl:text> </xsl:text>
        <span class="verbatim-pi-content">
          <xsl:value-of select="."/>
        </span>
      </xsl:if>
      <xsl:call-template name="close-pi"/>
    </div>
  </xsl:template>

  <xsl:template name="html-replace-entities">
    <xsl:param name="text"/>
    <xsl:param name="attrs"/>
    <xsl:variable name="tmp">
      <xsl:call-template name="replace-substring">
        <xsl:with-param name="from" select="'&gt;'"/>
        <xsl:with-param name="to" select="'&amp;gt;'"/>
        <xsl:with-param name="value">
          <xsl:call-template name="replace-substring">
            <xsl:with-param name="from" select="'&lt;'"/>
            <xsl:with-param name="to" select="'&amp;lt;'"/>
            <xsl:with-param name="value">
              <xsl:call-template name="replace-substring">
                <xsl:with-param name="from" select="'&amp;'"/>
                <xsl:with-param name="to" select="'&amp;amp;'"/>
                <xsl:with-param name="value" select="$text"/>
              </xsl:call-template>
            </xsl:with-param>
          </xsl:call-template>
        </xsl:with-param>
      </xsl:call-template>
    </xsl:variable>
    <xsl:choose>
      <!-- $text is an attribute value -->
      <xsl:when test="$attrs">
        <xsl:call-template name="replace-substring">
          <xsl:with-param name="from" select="'&#xA;'"/>
          <xsl:with-param name="to" select="'&amp;#xA;'"/>
          <xsl:with-param name="value">
            <xsl:call-template name="replace-substring">
              <xsl:with-param name="from" select="'&quot;'"/>
              <xsl:with-param name="to" select="'&amp;quot;'"/>
              <xsl:with-param name="value" select="$tmp"/>
            </xsl:call-template>
          </xsl:with-param>
        </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$tmp"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <!-- replace in $value substring $from with $to -->
  <xsl:template name="replace-substring">
    <xsl:param name="value"/>
    <xsl:param name="from"/>
    <xsl:param name="to"/>
    <xsl:choose>
      <xsl:when test="contains($value,$from)">
        <xsl:value-of select="substring-before($value,$from)"/>
        <xsl:value-of select="$to"/>
        <xsl:call-template name="replace-substring">
          <xsl:with-param name="value" select="substring-after($value,$from)"/>
          <xsl:with-param name="from" select="$from"/>
          <xsl:with-param name="to" select="$to"/>
        </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$value"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <!-- preformatted output: space as &nbsp;, tab as 8 &nbsp;
    nl as <br> -->
  <xsl:template name="preformatted-output">
    <xsl:param name="text"/>
    <xsl:value-of select="$text"/>

    <!--
      <xsl:call-template name="output-nl">
      <xsl:with-param name="text">
      <xsl:call-template name="replace-substring">
      <xsl:with-param name="value"
      select="translate($text,' ','&#xA0;')" />
      <xsl:with-param name="from" select="'&#9;'" />
      <xsl:with-param name="to" 
      select="'&#xA0;&#xA0;&#xA0;&#xA0;&#xA0;&#xA0;&#xA0;&#xA0;'" />
      </xsl:call-template>
      </xsl:with-param>
      </xsl:call-template>
    -->
  </xsl:template>

  <!-- output nl as <br> -->
  <xsl:template name="output-nl">
    <xsl:param name="text"/>
    <xsl:choose>
      <xsl:when test="contains($text,'&#xA;')">
        <xsl:value-of select="substring-before($text,'&#xA;')"/>
        <br/>
        <xsl:text>&#xA;</xsl:text>
        <xsl:call-template name="output-nl">
          <xsl:with-param name="text" select="substring-after($text,'&#xA;')"/>
        </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$text"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <!-- ============================================================= -->
  <!-- Templates used for marking up difference information          -->
  <!-- ============================================================= -->

  <xsl:template match="diff:*">
    <span class="{local-name()}">
      <xsl:apply-templates select="*"/>
    </span>
  </xsl:template>
  
  <!-- simpleminded highlighting and dynamic pop up showing which changeRef it applies to. -->

  <xsl:template name="if-diff">
    <xsl:choose>
      <xsl:when test="not(@diff)"/>
      <xsl:when test="not($show.changerefs) or (@changeref and not(matches(@changeref,$show.changerefs)))">
        <!-- show.changerefs parameter is null, which menas don't show any changes, just final result -->
        <xsl:choose>
          <xsl:when test="@diff='add'">
            <!--  -->
          </xsl:when>
          <xsl:when test="@diff='chg'">
            <!--  -->
          </xsl:when>
          <xsl:when test="@diff='del'">
            <xsl:attribute name="style" select="concat('display: none; ',@style)"/>
          </xsl:when>
          <xsl:when test="@diff='off'">
            <xsl:attribute name="style" select="concat('display: run-in; ',@style)"/>
          </xsl:when>
        </xsl:choose>
      </xsl:when>
      <!-- show.changerefs parameter is not null; show the change if its changeref matches the regex -->
      <!-- empty changeref matches everything -->
      <xsl:otherwise>
        <!-- it matched.  Add the value of the @diff parameter to the class attribute. -->
        <xsl:attribute name="class" select="normalize-space(concat(@diff,' ',@class))"/>
        <xsl:attribute name="onMouseOver">JavaScript:openit('<xsl:value-of select="@changeref"/>')</xsl:attribute>
        <xsl:attribute name="onMouseOut">JavaScript:hideit()</xsl:attribute>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>
  
  <xsl:template name="script">
    <script language="JavaScript"><![CDATA[
      // feel free to do something more elaborate with this.
      // w is a global for the 'pop up' window.
      var w = null;
      function openit(name) {
        if (!w) w = window.open("", "w", "width=100,height=20,scrollbars=no,titlebar=no");
        w.blur();
        w.moveTo(window.screenLeft+event.clientX,window.screenTop+event.clientY);
        w.document.open();
        w.document.write('<tt>'+name+'</tt>');
        w.focus()
}

function hideit() {  
  w.document.close();
  w.blur()
}
        ]]></script>
  </xsl:template>

</xsl:transform>
